/**
 * \file: mspin_lm_touch_adapter.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MySpin LayerManager Touch Event Adapter
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES thilobjoern.fickel@bosch-softtec.com
 *
 * \copyright: (c) 2015 Bosch SoftTec
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_lm_touch_adapter.h"
#include "mspin_lm_adapter_internal.h"
#include "mspin_logging.h"

#ifdef MSPIN_TOUCH_DELAY
#include "mspin_lm_delaytouch.h"
#endif

#include <linux/input.h> //BTN_TOUCH, etc.

static const U32 gFingerIdPointerButton1 = 0;  // same as finger id 0

static void mspin_lm_send_touch(mspin_layerManager_context_t *pLMContext)
{
    TouchEvent *pTouch = NULL;
    int size = 0;

    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(pWLCtx=%p) FATAL ERROR: LayerManager context is NULL -> abort function",
                __FUNCTION__, pLMContext);
        return;
    }

    if (!pLMContext->pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(pWLCtx=%p) FATAL ERROR: Wayland context is NULL -> abort function",
                __FUNCTION__, pLMContext);
        return;
    }

    mspin_wl_context_t *pWLCtx = pLMContext->pWLCtx;

    if (pWLCtx->touchInfo[0].valid)
    {
        pTouch = &pWLCtx->touchEv[0];
        if (pWLCtx->touchInfo[1].valid)
        {
            if (pLMContext->useTouchFiltering)
            {
                //In case of a gesture update make sure both times are updated
                if (pWLCtx->touchInfo[1].lastTimeAt > pWLCtx->touchInfo[0].lastTimeAt)
                {
                    pWLCtx->touchInfo[1].lastTimeAt = pWLCtx->touchInfo[0].lastTimeAt;
                }
                else
                {
                    pWLCtx->touchInfo[0].lastTimeAt =  pWLCtx->touchInfo[1].lastTimeAt;
                }
            }
            size = 2; // for gestures send both fingers, i.e. re-send the other finger/event
        }
        else
        {
            size = 1;
        }
    }
    else if (pWLCtx->touchInfo[1].valid)
    {
        pTouch = &pWLCtx->touchEv[1];
        size = 1;
    }

#ifdef MSPIN_TOUCH_DELAY
    mspin_lm_delayTouchEvents(pLMContext->pCoreHandle, pTouch, size);
#else
    mySpin_TouchEvents(pLMContext->pCoreHandle, pTouch, size);
#endif
}

void mspin_lm_touchHandleDown(void* data, struct wl_touch* touch, uint32_t serial, uint32_t wltime,
                struct wl_surface* surface, int32_t id, wl_fixed_t xw, wl_fixed_t yw)
{
    MSPIN_UNUSED(touch);
    MSPIN_UNUSED(serial);
    MSPIN_UNUSED(surface);

   int touch_x = (int)wl_fixed_to_double(xw);
   int touch_y = (int)wl_fixed_to_double(yw);

    WlSeat * pWlSeat = (WlSeat *)data;
    mspin_wl_context_t * pWLCtx = pWlSeat->p_wl_context;

    mspin_layerManager_context_t *pLMContext = (mspin_layerManager_context_t*) pWLCtx->p_lm_context;
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: p_lm_context context is NULL",
                __FUNCTION__);
        return;
    }

    if (pWLCtx)
    {
        if ((id >= 0) && (id < MSPIN_WL_MAX_TOUCHPOINTS))
        {
            // workaround for driver sometimes detecting wrong finger up SWGIII-3744:
            if ((id == 1) && (wltime - pWLCtx->lasttimeup1 < 20) && pWLCtx->touchInfo[0].valid)
            {
                // send finger id 0 up:
                pWLCtx->touchEv[0].event = eTOUCHTYPE_Up;
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s() touch up for id=0 Correction!", __FUNCTION__);
                mspin_lm_send_touch(pLMContext);
                pWLCtx->touchInfo[0].valid = FALSE;
            }

            pWLCtx->touchEv[id].xPosition = (U16)touch_x;
            pWLCtx->touchEv[id].yPosition = (U16)touch_y;
            pWLCtx->touchEv[id].event = eTOUCHTYPE_Down;
            pWLCtx->touchInfo[id].valid = TRUE;
            pWLCtx->touchInfo[id].lastTimeAt = wltime;

            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s() touch down %d, %d for id=%d #=%d",
                    __FUNCTION__, touch_x, touch_y, id,
                    (pWLCtx->touchInfo[0].valid && pWLCtx->touchInfo[1].valid) ? 2 : 1);
            mspin_lm_send_touch(pLMContext);
            pWLCtx->touchEv[id].event = eTOUCHTYPE_Moved; // avoid storing down, since it might be resent
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityWarn,
                    "%s() touch down %d, %d for unhandled id=%d",
                    __FUNCTION__, touch_x, touch_y, id);
        }
    }
}


void mspin_lm_touchHandleUp(void* data, struct wl_touch* touch, uint32_t serial, uint32_t wltime, int32_t id)
{
    MSPIN_UNUSED(touch);
    MSPIN_UNUSED(serial);

    WlSeat * pWlSeat = (WlSeat *)data;
    mspin_wl_context_t * pWLCtx = pWlSeat->p_wl_context;

    mspin_layerManager_context_t *pLMContext = (mspin_layerManager_context_t*) pWLCtx->p_lm_context;

    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: LayerManager context is NULL", __FUNCTION__);
        return;
    }

    if (pWLCtx)
    {
        if ((id >= 0) && (id < MSPIN_WL_MAX_TOUCHPOINTS))
        {
            pWLCtx->touchEv[id].event = eTOUCHTYPE_Up;
            pWLCtx->touchInfo[id].lastTimeAt = 0; //reset time
            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s() touch up for id=%d #=%d at time=%d",
                    __FUNCTION__, id,
                    (pWLCtx->touchInfo[0].valid && pWLCtx->touchInfo[1].valid) ? 2 : 1, wltime);
            mspin_lm_send_touch(pLMContext);
            pWLCtx->touchInfo[id].valid = FALSE;
            if (id == 1)
            {
                pWLCtx->lasttimeup1 = wltime;
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityWarn,
                    "%s() touch up for unhandled id=%d", __FUNCTION__, id);
        }
    }
}

void mspin_lm_touchHandleMotion(void* data, struct wl_touch* touch, uint32_t wltime, int32_t id,
                  wl_fixed_t xw, wl_fixed_t yw)
{
    MSPIN_UNUSED(touch);

    //ToDo: Use U16 as type instead of int
    int touch_x = (int)wl_fixed_to_double(xw); //ToDo: use wl_fixed_to_int instead -> to be checked
    int touch_y = (int)wl_fixed_to_double(yw); //ToDo: use wl_fixed_to_int instead
    int distance = 0;

    WlSeat * pWlSeat = (WlSeat *)data;
    mspin_wl_context_t * pWLCtx = pWlSeat->p_wl_context;

    mspin_layerManager_context_t *pLMContext = (mspin_layerManager_context_t*) pWLCtx->p_lm_context;

    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: LayerManager context is NULL", __FUNCTION__);
        return;
    }

    if (pWLCtx)
    {
        if ((id >= 0) && (id < MSPIN_WL_MAX_TOUCHPOINTS))
        {
            //Filter touch events if enabled
            if (pLMContext->useTouchFiltering && pWLCtx->touchInfo[id].valid)
            {
                distance = abs(touch_x - pWLCtx->touchEv[id].xPosition) + abs(touch_y - pWLCtx->touchEv[id].yPosition);
                if (distance < MSPIN_MAX_TOUCH_MOVE_DISTANCE_IGNORED)
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(id=%d, x=%d, y=%d) position is only %d pixels different => ignore",
                            __FUNCTION__, id, touch_x, touch_y, distance);
                    return;
                }

                if (wltime - pWLCtx->touchInfo[id].lastTimeAt < MSPIN_LM_MINIMUM_TOUCH_INTERVAL_IN_MS)
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(wltime=%d, id=%d, x=%d, y=%d) last event is only %dms old (< %dms) => ignore",
                            __FUNCTION__, wltime, id, touch_x, touch_y, wltime - pWLCtx->touchInfo[id].lastTimeAt,
                            MSPIN_LM_MINIMUM_TOUCH_INTERVAL_IN_MS);
                    return;
                }

                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(id=%d, x=%d, y=%d) position is %d >= %d pixels different => forward",
                        __FUNCTION__, id, touch_x, touch_y, distance, MSPIN_MAX_TOUCH_MOVE_DISTANCE_IGNORED);
            }

            pWLCtx->touchEv[id].xPosition = (U16)touch_x;
            pWLCtx->touchEv[id].yPosition = (U16)touch_y;
            pWLCtx->touchEv[id].event = eTOUCHTYPE_Moved;
            pWLCtx->touchInfo[id].lastTimeAt = wltime;
            // workaround for driver sometimes detecting wrong finger up SWGIII-3744:
            if (!pWLCtx->touchInfo[id].valid)
            {
                pWLCtx->touchEv[id].event = eTOUCHTYPE_Down;
                mspin_log_printLn(eMspinVerbosityInfo,
                        "%s() touch down for id=%d Correction!", __FUNCTION__, id);
            }
            pWLCtx->touchInfo[id].valid = TRUE;
            mspin_log_printLn(eMspinVerbosityDebug, "%s() touch moved %d, %d for id=%d #=%d",
                    __FUNCTION__, touch_x, touch_y, id,
                    (pWLCtx->touchInfo[0].valid &&  pWLCtx->touchInfo[1].valid) ? 2 : 1);
            mspin_lm_send_touch(pLMContext);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityWarn,
                    "%s() touch moved %d, %d for unhandled id=%d",
                    __FUNCTION__, touch_x, touch_y, id);
        }
    }
}

void mspin_lm_touchHandleFrame(void* data, struct wl_touch* touch)
{
    MSPIN_UNUSED(data);
    MSPIN_UNUSED(touch);
}

void mspin_lm_touchHandleCancel(void* data, struct wl_touch* touch)
{
    MSPIN_UNUSED(data);
    MSPIN_UNUSED(touch);

    mspin_log_printLn(eMspinVerbosityWarn,
            "%s() WARNING: touch cancel not handled",
            __FUNCTION__);
}


void mspin_lm_pointerHandleEnter(void* data, struct wl_pointer* wlPointer, uint32_t serial,
                   struct wl_surface* wlSurface, wl_fixed_t sx, wl_fixed_t sy)
{
    MSPIN_UNUSED(data);
    MSPIN_UNUSED(wlPointer);
    MSPIN_UNUSED(serial);
    MSPIN_UNUSED(wlSurface);
    MSPIN_UNUSED(sx);
    MSPIN_UNUSED(sy);
    mspin_log_printLn(eMspinVerbosityDebug, "%s() Not implemented", __FUNCTION__);
}


void mspin_lm_pointerHandleLeave(void* data, struct wl_pointer* wlPointer, uint32_t serial,
                   struct wl_surface* wlSurface)
{
    MSPIN_UNUSED(data);
    MSPIN_UNUSED(wlPointer);
    MSPIN_UNUSED(serial);
    MSPIN_UNUSED(wlSurface);
    mspin_log_printLn(eMspinVerbosityDebug, "%s() Not implemented", __FUNCTION__);
}

void mspin_lm_pointerHandleMotion(void* data, struct wl_pointer* wlPointer, uint32_t timestamp,
                    wl_fixed_t sx, wl_fixed_t sy)
{
    MSPIN_UNUSED(wlPointer);

    int pointerX = (int)wl_fixed_to_double(sx);
    int pointerY = (int)wl_fixed_to_double(sy);
    int distance = 0;

    WlSeat * pWlSeat = (WlSeat *)data;
    mspin_wl_context_t * pWLCtx = pWlSeat->p_wl_context;

    mspin_layerManager_context_t *pLMContext = (mspin_layerManager_context_t*) pWLCtx->p_lm_context;
    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: p_lm_context context is NULL",
                __FUNCTION__);
        return;
    }

    if (pWLCtx)
    {
        if (pWLCtx->pointerButtonState != 0)
        {
            //Filter pointer events if enabled
            if (pLMContext->usePointerFiltering && !pWLCtx->pointerButtonDownToBeSent)
            {
                distance = abs(pointerX - pWLCtx->lastSentPointerX) + abs(pointerY - pWLCtx->lastSentPointerY);
                if (0 == distance)
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(x=%d, y=%d) same position => ignore",
                            __FUNCTION__, pointerX, pointerY);
                    return;
                }
                else if (distance < MSPIN_MAX_POINTER_MOVE_DISTANCE_IGNORED)
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(x=%d, y=%d) position is only %d pixels different => ignore",
                            __FUNCTION__, pointerX, pointerY, distance);
                    return;
                }

                if (timestamp - pWLCtx->lastTimePointerEventSent < MSPIN_LM_MINIMUM_POINTER_INTERVAL_IN_MS)
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(time=%d, x=%d, y=%d) last event is only %dms old (< %dms) => ignore",
                            __FUNCTION__, timestamp, pointerX, pointerY, timestamp - pWLCtx->lastTimePointerEventSent,
                            MSPIN_LM_MINIMUM_POINTER_INTERVAL_IN_MS);
                    return;
                }

                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(x=%d, y=%d) position is %d >= %d pixels different => forward",
                        __FUNCTION__, pointerX, pointerY, distance, MSPIN_MAX_POINTER_MOVE_DISTANCE_IGNORED);
            }

            TouchEvent touchEv;
            touchEv.xPosition = pointerX;
            touchEv.yPosition = pointerY;
            pWLCtx->lastSentPointerX = pointerX;
            pWLCtx->lastSentPointerY = pointerY;
            pWLCtx->lastTimePointerEventSent = timestamp;
            touchEv.fingerId = gFingerIdPointerButton1;
            touchEv.event = eTOUCHTYPE_Moved;
            if (pWLCtx->pointerButtonDownToBeSent) // now we have a valid position and can sent the Down event
            {
                touchEv.event = eTOUCHTYPE_Down;
            	pWLCtx->pointerButtonDownToBeSent = FALSE;
 			}

            mySpin_TouchEvents(pLMContext->pCoreHandle, &touchEv, 1);
            mspin_log_printLn(eMspinVerbosityInfo, "%s() pointer %s %d, %d for id=%d at time=%d",
                    __FUNCTION__, (touchEv.event == eTOUCHTYPE_Moved) ? "moved" : "button down",
                    touchEv.xPosition, touchEv.yPosition, gFingerIdPointerButton1, timestamp);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s() pointer moved %d, %d for id=%d but button not yet pressed",
                    __FUNCTION__, pointerX, pointerY, gFingerIdPointerButton1);
        }

        //Store always the coordinates. They will be used in mspin_lm_pointerHandleButton because this event does not
        // have any coordinates
        pWLCtx->lastPointerX = pointerX;
        pWLCtx->lastPointerY = pointerY;
    }
}

void mspin_lm_pointerHandleButton(void* data, struct wl_pointer* wlPointer, uint32_t serial,
                    uint32_t timestamp, uint32_t button, uint32_t state)
{
    MSPIN_UNUSED(wlPointer);
    MSPIN_UNUSED(serial);
    MSPIN_UNUSED(timestamp);

    WlSeat * pWlSeat = (WlSeat *)data;
    mspin_wl_context_t * pWLCtx = pWlSeat->p_wl_context;

    mspin_layerManager_context_t *pLMContext = (mspin_layerManager_context_t*) pWLCtx->p_lm_context;

    if (!pLMContext)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: LayerManager context is NULL", __FUNCTION__);
        return;
    }

    if (pWLCtx)
    {
        switch (button)
        {
        case BTN_TOUCH: // fall through
        case BTN_LEFT:
        {
            if (state != 0)
            {
                state = 1;
            }
            // send touch event with saved position if state for TOUCH/LEFT button changed
            if (state != pWLCtx->pointerButtonState)
            {
                TouchEvent touchEv;
                touchEv.xPosition = (U16)pWLCtx->lastPointerX;
                touchEv.yPosition = (U16)pWLCtx->lastPointerY;
                touchEv.fingerId = gFingerIdPointerButton1;

                if (state == 0)
                {
                    //Pointer button up event
                    touchEv.event = eTOUCHTYPE_Up;
                    pWLCtx->lastTimePointerEventSent = 0; //clear time
                    mspin_log_printLn(eMspinVerbosityInfo, "%s() pointer button up %d, %d for id=%d at time=%d",
                            __FUNCTION__, touchEv.xPosition, touchEv.yPosition, gFingerIdPointerButton1, timestamp);

                    pWLCtx->lastSentPointerX = touchEv.xPosition;
                    pWLCtx->lastSentPointerY = touchEv.yPosition;

                    mySpin_TouchEvents(pLMContext->pCoreHandle, &touchEv, 1);
                }
                else
                {
                	// remember the Down, since position is missing and saved position is not yet correct
                	// Attention: the position that is just sent before by the touch driver/layermanager
                	//            is still the position from the last touch, i.e. invalid!
                	pWLCtx->pointerButtonDownToBeSent = TRUE;
                }

                pWLCtx->pointerButtonState = state;
            }
            break;
        }
        case BTN_RIGHT: // fall through
        case BTN_MIDDLE: // fall through
        default:
            // nothing
            break;
        }
    }
}


void mspin_lm_pointerHandleAxis(void* data, struct wl_pointer* wlPointer, uint32_t timestamp,
                  uint32_t axis, wl_fixed_t value)
{
    MSPIN_UNUSED(data);
    MSPIN_UNUSED(wlPointer);
    MSPIN_UNUSED(timestamp);
    MSPIN_UNUSED(axis);
    MSPIN_UNUSED(value);

    mspin_log_printLn(eMspinVerbosityDebug, "%s() Not implemented", __FUNCTION__);
}

void mspin_lm_pointerHandleFrame(void *data, struct wl_pointer *wlPointer)
{
    MSPIN_UNUSED(data);
    MSPIN_UNUSED(wlPointer);

    mspin_log_printLn(eMspinVerbosityDebug, "%s() Not implemented", __FUNCTION__);
}

void mspin_lm_pointerHandleAxisSource(void *data, struct wl_pointer *wlPointer, uint32_t source)
    {
    MSPIN_UNUSED(data);
    MSPIN_UNUSED(wlPointer);
    MSPIN_UNUSED(source);

    mspin_log_printLn(eMspinVerbosityDebug, "%s() Not implemented", __FUNCTION__);
}

void mspin_lm_pointerHandleAxisStop(void *data, struct wl_pointer *wlPointer,
            uint32_t timestamp, uint32_t axis)
{
    MSPIN_UNUSED(data);
    MSPIN_UNUSED(wlPointer);
    MSPIN_UNUSED(timestamp);
    MSPIN_UNUSED(axis);

    mspin_log_printLn(eMspinVerbosityDebug, "%s() Not implemented", __FUNCTION__);
}

void mspin_lm_pointerHandleAxisDiscrete(void *data, struct wl_pointer *wlPointer,
            uint32_t axis, int32_t discrete)
{
    MSPIN_UNUSED(data);
    MSPIN_UNUSED(wlPointer);
    MSPIN_UNUSED(axis);
    MSPIN_UNUSED(discrete);

    mspin_log_printLn(eMspinVerbosityDebug, "%s() Not implemented", __FUNCTION__);
}
